/**
 * HRMS UI Enhancer
 * Automatically applies enhanced CSS classes to existing elements
 * and adds modern UI features
 */

(function ($) {
    'use strict';

    const HRM_UI_Enhancer = {

        /**
         * Initialize the enhancer
         */
        init: function () {
            console.log('HRM UI Enhancer initialized');
            this.enhanceTables();
            this.enhanceButtons();
            this.enhanceBadges();
            this.enhanceForms();
            this.addStaggerAnimations();
            this.addIconAnimations();
            this.initDropdowns();
            this.initTooltips();
            this.enhancePagination();
        },

        /**
         * Enhance tables with modern styling
         */
        enhanceTables: function () {
            // Wrap tables that aren't already wrapped
            $('.wp-list-table, table.widefat, .hrm-wrap table').not('.hrm-table').each(function () {
                const $table = $(this);

                // Skip if already enhanced
                if ($table.closest('.hrm-table-wrapper').length) {
                    return;
                }

                // Add enhanced class
                $table.addClass('hrm-table');

                // Wrap in table wrapper if not already wrapped
                if (!$table.parent().hasClass('hrm-table-wrapper')) {
                    $table.wrap('<div class="hrm-table-wrapper"></div>');
                }

                // Add hover effects
                $table.find('tbody tr').hover(
                    function () {
                        $(this).css('transform', 'translateX(2px)');
                    },
                    function () {
                        $(this).css('transform', 'translateX(0)');
                    }
                );
            });
        },

        /**
         * Enhance buttons
         */
        enhanceButtons: function () {
            // Primary buttons
            $('.button-primary, input[type="submit"].button-primary').each(function () {
                const $btn = $(this);
                if (!$btn.hasClass('hrm-btn-primary') && !$btn.hasClass('enhanced')) {
                    $btn.addClass('hrm-btn-primary enhanced');
                }
            });

            // Secondary buttons
            $('.button-secondary, .button').not('.button-primary').each(function () {
                const $btn = $(this);
                if (!$btn.hasClass('hrm-btn-outline') && !$btn.hasClass('enhanced')) {
                    $btn.addClass('hrm-btn-outline enhanced');
                }
            });

            // Add ripple effect to all buttons
            this.addRippleEffect();

            // Add loading state on form submit
            $('form').on('submit', function () {
                $(this).find('input[type="submit"], button[type="submit"]').addClass('loading');
            });
        },

        /**
         * Add ripple effect to buttons
         */
        addRippleEffect: function () {
            const rippleClass = 'hrm-ripple-effect';

            $(document).on('click', '.hrm-btn, .hrm-btn-primary, .hrm-btn-outline, .hrm-btn-danger, .hrm-btn-icon, .hrm-icon-btn', function (e) {
                const $btn = $(this);

                // Skip if disabled
                if ($btn.prop('disabled') || $btn.hasClass('disabled')) return;

                // Remove existing ripple
                $btn.find('.' + rippleClass).remove();

                // Create ripple
                const $ripple = $('<span class="' + rippleClass + '"></span>');
                $btn.css('position', 'relative').css('overflow', 'hidden');

                // Calculate position
                const rect = $btn[0].getBoundingClientRect();
                const size = Math.max(rect.width, rect.height) * 2;

                $ripple.css({
                    width: size + 'px',
                    height: size + 'px',
                    left: (e.clientX - rect.left - size / 2) + 'px',
                    top: (e.clientY - rect.top - size / 2) + 'px',
                    position: 'absolute',
                    borderRadius: '50%',
                    background: 'rgba(255, 255, 255, 0.4)',
                    transform: 'scale(0)',
                    animation: 'ripple 0.5s ease-out',
                    pointerEvents: 'none'
                });

                $btn.append($ripple);

                // Remove after animation
                setTimeout(function () {
                    $ripple.remove();
                }, 500);
            });

            // Add CSS animation for ripple if not exists
            if (!$('#hrm-ripple-styles').length) {
                $('head').append(`
                    <style id="hrm-ripple-styles">
                        @keyframes ripple {
                            to {
                                transform: scale(2);
                                opacity: 0;
                            }
                        }
                        .hrm-ripple-effect {
                            animation: ripple 0.5s ease-out forwards;
                        }
                    </style>
                `);
            }
        },

        /**
         * Add hover animations to icon buttons
         */
        addIconAnimations: function () {
            // Icon button hover scale
            $(document).on('mouseenter', '.hrm-btn-icon, .hrm-icon-btn', function () {
                $(this).find('.dashicons, .hrm-icon').css({
                    'transform': 'scale(1.15)',
                    'transition': 'transform 0.15s cubic-bezier(0.34, 1.56, 0.64, 1)'
                });
            }).on('mouseleave', '.hrm-btn-icon, .hrm-icon-btn', function () {
                $(this).find('.dashicons, .hrm-icon').css({
                    'transform': 'scale(1)'
                });
            });

            // Navigation icon animations
            $(document).on('mouseenter', '.hrm-tab-btn, .hrm-nav-menu li', function () {
                $(this).find('.dashicons, .hrm-icon').css({
                    'transform': 'translateX(3px) scale(1.1)',
                    'transition': 'transform 0.2s cubic-bezier(0.34, 1.56, 0.64, 1)'
                });
            }).on('mouseleave', '.hrm-tab-btn, .hrm-nav-menu li', function () {
                $(this).find('.dashicons, .hrm-icon').css({
                    'transform': 'translateX(0) scale(1)'
                });
            });

            // Stat card icon pulse on hover
            $(document).on('mouseenter', '.hrm-stat-card, .hrm-mini-stat', function () {
                $(this).find('.stat-header .dashicons, .stat-header .hrm-icon').css({
                    'transform': 'scale(1.15) rotate(-5deg)',
                    'transition': 'transform 0.3s cubic-bezier(0.175, 0.885, 0.32, 1.275)'
                });
            }).on('mouseleave', '.hrm-stat-card, .hrm-mini-stat', function () {
                $(this).find('.stat-header .dashicons, .stat-header .hrm-icon').css({
                    'transform': 'scale(1) rotate(0deg)'
                });
            });
        },

        /**
         * Enhance status badges
         */
        enhanceBadges: function () {
            const statusMap = {
                'present': 'hrm-badge-success',
                'active': 'hrm-badge-success',
                'completed': 'hrm-badge-success',
                'approved': 'hrm-badge-success',

                'absent': 'hrm-badge-danger',
                'inactive': 'hrm-badge-danger',
                'rejected': 'hrm-badge-danger',
                'failed': 'hrm-badge-danger',

                'pending': 'hrm-badge-warning',
                'in-progress': 'hrm-badge-warning',
                'in progress': 'hrm-badge-warning',
                'processing': 'hrm-badge-warning',

                'info': 'hrm-badge-info',
                'information': 'hrm-badge-info',
            };

            // Find potential status elements
            $('.hrm-wrap span, .hrm-wrap .status, .hrm-notice').each(function () {
                const $el = $(this);
                const text = $el.text().toLowerCase().trim();

                // Check if text matches a status
                if (statusMap[text] && !$el.hasClass('hrm-badge')) {
                    $el.addClass('hrm-badge ' + statusMap[text]);
                }
            });

            // Convert WordPress notices to enhanced badges
            $('.hrm-notice').each(function () {
                const $notice = $(this);
                if ($notice.hasClass('error')) {
                    $notice.addClass('hrm-badge-danger');
                } else if ($notice.hasClass('success')) {
                    $notice.addClass('hrm-badge-success');
                } else if ($notice.hasClass('warning')) {
                    $notice.addClass('hrm-badge-warning');
                }
            });
        },

        /**
         * Enhance form elements
         */
        enhanceForms: function () {
            // Enhance inputs
            $('.hrm-wrap input[type="text"], .hrm-wrap input[type="email"], .hrm-wrap input[type="password"], .hrm-wrap input[type="number"], .hrm-wrap input[type="date"], .hrm-wrap input[type="time"]').each(function () {
                const $input = $(this);
                if (!$input.hasClass('hrm-input') && !$input.hasClass('enhanced')) {
                    $input.addClass('hrm-input enhanced');
                }
            });

            // Enhance selects
            $('.hrm-wrap select').each(function () {
                const $select = $(this);
                if (!$select.hasClass('hrm-select') && !$select.hasClass('enhanced')) {
                    $select.addClass('hrm-select enhanced');
                }
            });

            // Enhance textareas
            $('.hrm-wrap textarea').each(function () {
                const $textarea = $(this);
                if (!$textarea.hasClass('hrm-textarea') && !$textarea.hasClass('enhanced')) {
                    $textarea.addClass('hrm-textarea enhanced');
                }
            });

            // Add focus animations
            $('.hrm-input, .hrm-select, .hrm-textarea').on('focus', function () {
                $(this).parent().addClass('focused');
            }).on('blur', function () {
                $(this).parent().removeClass('focused');
            });

            // Add required indicator to labels
            $('input[required], select[required], textarea[required]').each(function () {
                const $field = $(this);
                const $label = $('label[for="' + $field.attr('id') + '"]');
                if ($label.length && !$label.hasClass('required')) {
                    $label.addClass('required');
                }
            });
        },

        /**
         * Add stagger animations to stat cards
         */
        addStaggerAnimations: function () {
            // Add stagger animation to stat rows
            $('.hrm-stats-row').addClass('hrm-stagger');

            // Trigger count-up animation for numbers
            $('.stat-number, .value').each(function () {
                const $el = $(this);
                const target = parseInt($el.text().replace(/,/g, ''));

                if (!isNaN(target) && target > 0) {
                    $el.prop('Counter', 0).animate({
                        Counter: target
                    }, {
                        duration: 1500,
                        easing: 'swing',
                        step: function (now) {
                            $el.text(Math.ceil(now).toLocaleString());
                        }
                    });
                }
            });
        },

        /**
         * Initialize dropdown menus
         */
        initDropdowns: function () {
            $('.hrm-dropdown-toggle').on('click', function (e) {
                e.preventDefault();
                e.stopPropagation();

                const $dropdown = $(this).closest('.hrm-dropdown');
                const isActive = $dropdown.hasClass('active');

                // Close all other dropdowns
                $('.hrm-dropdown').removeClass('active');

                // Toggle this dropdown
                if (!isActive) {
                    $dropdown.addClass('active');
                }
            });

            // Close dropdowns when clicking outside
            $(document).on('click', function (e) {
                if (!$(e.target).closest('.hrm-dropdown').length) {
                    $('.hrm-dropdown').removeClass('active');
                }
            });

            // Close on escape key
            $(document).on('keydown', function (e) {
                if (e.key === 'Escape') {
                    $('.hrm-dropdown').removeClass('active');
                }
            });
        },

        /**
         * Initialize tooltips
         */
        initTooltips: function () {
            // Add data-tooltip attributes to elements that need them
            $('[title]').each(function () {
                const $el = $(this);
                const title = $el.attr('title');
                if (title && !$el.attr('data-tooltip')) {
                    $el.attr('data-tooltip', title);
                    $el.removeAttr('title');
                }
            });
        },

        /**
         * Enhance pagination
         */
        enhancePagination: function () {
            // Wrap pagination if needed
            $('.tablenav-pages').each(function () {
                const $pages = $(this);
                if (!$pages.hasClass('hrm-pagination')) {
                    $pages.addClass('hrm-pagination');

                    // Enhance page links
                    $pages.find('a.page-numbers, span.page-numbers').addClass('hrm-pagination-btn');
                    $pages.find('.current').addClass('active');
                }
            });
        },

        /**
         * Add skeleton loaders
         */
        addSkeletonLoaders: function (container) {
            const $container = $(container);
            const skeletonHTML = `
                <div class="hrm-skeleton-container">
                    <div class="hrm-skeleton hrm-skeleton-title"></div>
                    <div class="hrm-skeleton hrm-skeleton-text"></div>
                    <div class="hrm-skeleton hrm-skeleton-text"></div>
                    <div class="hrm-skeleton hrm-skeleton-text"></div>
                </div>
            `;
            $container.html(skeletonHTML);
        },

        /**
         * Remove skeleton loaders
         */
        removeSkeletonLoaders: function (container) {
            $(container).find('.hrm-skeleton-container').remove();
        },

        /**
         * Show toast notification
         */
        showToast: function (message, type = 'success', duration = 3000) {
            const toastHTML = `
                <div class="hrm-toast hrm-toast-${type} hrm-animate-slideInRight">
                    <div class="hrm-toast-content">
                        <span class="hrm-toast-icon dashicons dashicons-${type === 'success' ? 'yes' : 'warning'}"></span>
                        <span class="hrm-toast-message">${message}</span>
                    </div>
                    <button class="hrm-toast-close">&times;</button>
                </div>
            `;

            const $toast = $(toastHTML);

            // Add to body
            if (!$('#hrm-toast-container').length) {
                $('body').append('<div id="hrm-toast-container"></div>');
            }

            $('#hrm-toast-container').append($toast);

            // Auto-remove
            setTimeout(function () {
                $toast.fadeOut(300, function () {
                    $(this).remove();
                });
            }, duration);

            // Manual close
            $toast.find('.hrm-toast-close').on('click', function () {
                $toast.fadeOut(300, function () {
                    $(this).remove();
                });
            });
        },

        /**
         * Confirm dialog
         */
        confirm: function (message, callback) {
            const confirmHTML = `
                <div class="hrm-modal hrm-animate-fadeIn" style="display: flex;">
                    <div class="hrm-modal-overlay"></div>
                    <div class="hrm-modal-content hrm-animate-scaleIn">
                        <div class="hrm-modal-header">
                            <h3>Confirm Action</h3>
                        </div>
                        <div class="hrm-modal-body">
                            <p>${message}</p>
                        </div>
                        <div class="hrm-modal-footer">
                            <button class="hrm-btn-outline hrm-modal-cancel">Cancel</button>
                            <button class="hrm-btn-primary hrm-modal-confirm">Confirm</button>
                        </div>
                    </div>
                </div>
            `;

            const $modal = $(confirmHTML);
            $('body').append($modal);

            $modal.find('.hrm-modal-confirm').on('click', function () {
                $modal.fadeOut(200, function () {
                    $(this).remove();
                });
                if (callback) callback(true);
            });

            $modal.find('.hrm-modal-cancel, .hrm-modal-overlay').on('click', function () {
                $modal.fadeOut(200, function () {
                    $(this).remove();
                });
                if (callback) callback(false);
            });
        },

        /**
         * Add loading overlay
         */
        showLoading: function (container, message = 'Loading...') {
            const loadingHTML = `
                <div class="hrm-loading-overlay">
                    <div class="hrm-loading-spinner"></div>
                    <div class="hrm-loading-message">${message}</div>
                </div>
            `;
            $(container).css('position', 'relative').append(loadingHTML);
        },

        /**
         * Remove loading overlay
         */
        hideLoading: function (container) {
            $(container).find('.hrm-loading-overlay').fadeOut(200, function () {
                $(this).remove();
            });
        }
    };

    // Auto-initialize when document is ready
    $(document).ready(function () {
        HRM_UI_Enhancer.init();
    });

    // Also re-enhance after AJAX updates
    $(document).ajaxComplete(function () {
        setTimeout(function () {
            HRM_UI_Enhancer.enhanceTables();
            HRM_UI_Enhancer.enhanceButtons();
            HRM_UI_Enhancer.enhanceBadges();
            HRM_UI_Enhancer.enhanceForms();
        }, 100);
    });

    // Make globally available
    window.HRM_UI_Enhancer = HRM_UI_Enhancer;

})(jQuery);
